<?php
session_start();
require_once "../includes/role_helpers.php";
requireAdmin();
require_once "../config/config.php";

$conn = getDbConnection();

$error_msg = "";
$success_msg = "";

// Handle delete action
if (isset($_GET['delete_id'])) {
    $delete_id = intval($_GET['delete_id']);
    $stmt = $conn->prepare("DELETE FROM branches WHERE branch_id = ?");
    $stmt->bind_param("i", $delete_id);
    if ($stmt->execute()) {
        $success_msg = "Branch deleted successfully.";
    } else {
        $error_msg = "Error deleting branch.";
    }
    $stmt->close();
}

// Handle add/edit form submit
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $branch_id = intval($_POST['branch_id'] ?? 0);
    $branch_name = trim($_POST['branch_name'] ?? '');
    $branch_address = trim($_POST['branch_address'] ?? '');
    $branch_phone = trim($_POST['branch_phone'] ?? '');

    if ($branch_name === '') {
        $error_msg = "Branch name is required.";
    } else {
        if ($branch_id > 0) {
            // Update existing branch
            $stmt = $conn->prepare("UPDATE branches SET branch_name = ?, address = ?, phone = ? WHERE branch_id = ?");
            $stmt->bind_param("sssi", $branch_name, $branch_address, $branch_phone, $branch_id);
            if ($stmt->execute()) {
                $success_msg = "Branch updated successfully.";
            } else {
                $error_msg = "Error updating branch: " . $stmt->error;
            }
            $stmt->close();
        } else {
            // Insert new branch
            $stmt = $conn->prepare("INSERT INTO branches (branch_name, address, phone) VALUES (?, ?, ?)");
            $stmt->bind_param("sss", $branch_name, $branch_address, $branch_phone);
            if ($stmt->execute()) {
                $success_msg = "Branch added successfully.";
            } else {
                $error_msg = "Error adding branch: " . $stmt->error;
            }
            $stmt->close();
        }
    }
}

// Fetch all branches for listing
$result = $conn->query("SELECT * FROM branches ORDER BY branch_name");
$branches = [];
if ($result) {
    $branches = $result->fetch_all(MYSQLI_ASSOC);
    $result->close();
}

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Branch Management</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
</head>
<body>
<div class="container py-4">
    <h2>Branch Management</h2>

    <?php if ($success_msg): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success_msg) ?></div>
    <?php endif; ?>
    <?php if ($error_msg): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($error_msg) ?></div>
    <?php endif; ?>

    <form method="post" class="mb-4" id="branch-form">
        <input type="hidden" name="branch_id" id="branch_id" value="0" />
        <div class="mb-3">
            <label for="branch_name" class="form-label">Branch Name *</label>
            <input type="text" id="branch_name" name="branch_name" class="form-control" required />
        </div>
        <div class="mb-3">
            <label for="branch_address" class="form-label">Address</label>
            <textarea id="branch_address" name="branch_address" class="form-control"></textarea>
        </div>
        <div class="mb-3">
            <label for="branch_phone" class="form-label">Phone</label>
            <input type="text" id="branch_phone" name="branch_phone" class="form-control" />
        </div>
        <button type="submit" class="btn btn-primary">Save Branch</button>
        <button type="button" class="btn btn-secondary" onclick="resetForm()">Clear</button>
    </form>

    <h3>Existing Branches</h3>
    <table class="table table-bordered">
        <thead>
            <tr>
                <th>Branch Name</th>
                <th>Address</th>
                <th>Phone</th>
                <th style="width:120px;">Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php if (empty($branches)): ?>
                <tr><td colspan="4" class="text-center">No branches found.</td></tr>
            <?php else: ?>
                <?php foreach ($branches as $branch): ?>
                    <tr>
                        <td><?= htmlspecialchars($branch['branch_name']) ?></td>
                        <td><?= nl2br(htmlspecialchars($branch['address'])) ?></td>
                        <td><?= htmlspecialchars($branch['phone']) ?></td>
                        <td>
                            <button class="btn btn-sm btn-warning me-1" onclick='populateForm(<?= json_encode($branch) ?>)'>Edit</button>
                            <a href="?delete_id=<?= $branch['branch_id'] ?>" class="btn btn-sm btn-danger" onclick="return confirm('Delete this branch?');">Delete</a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            <?php endif; ?>
        </tbody>
    </table>
</div>

<script>
function populateForm(branch) {
    document.getElementById('branch_id').value = branch.branch_id;
    document.getElementById('branch_name').value = branch.branch_name;
    document.getElementById('branch_address').value = branch.address;
    document.getElementById('branch_phone').value = branch.phone;
    window.scrollTo(0,0);
}
function resetForm() {
    document.getElementById('branch-form').reset();
    document.getElementById('branch_id').value = 0;
}
</script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
