<?php
require_once "../config/config.php";
require_once "../includes/auth_check.php";
requireAdmin();

$conn = getDbConnection();

$id = $_GET['id'] ?? null;
if (!$id) {
    header('Location: manage_users.php');
    exit;
}

$username = $full_name = $email = $role = "";
$branch_id = 0;
$username_err = $email_err = $password_err = "";
$success_message = "";

// Fetch branches for dropdown
$branch_result = $conn->query("SELECT branch_id, branch_name FROM branches ORDER BY branch_name");
$branches = [];
if ($branch_result) {
    $branches = $branch_result->fetch_all(MYSQLI_ASSOC);
}

// Fetch existing user data including branch
$stmt = $conn->prepare("SELECT username, full_name, email, role, branch_id FROM users WHERE user_id = ?");
$stmt->bind_param("i", $id);
$stmt->execute();
$stmt->bind_result($username, $full_name, $email, $role, $branch_id);
if (!$stmt->fetch()) {
    $stmt->close();
    header('Location: manage_users.php');
    exit;
}
$stmt->close();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username_new = trim($_POST['username']);
    $full_name_new = trim($_POST['full_name']);
    $email_new = trim($_POST['email']);
    $role_new = $_POST['role'];
    $branch_id_new = intval($_POST['branch_id']);
    $password_new = trim($_POST['password']);

    $valid = true;
    if (empty($username_new)) { $username_err = "Username cannot be empty."; $valid = false; }
    if (empty($email_new)) { $email_err = "Email cannot be empty."; $valid = false; }
    if (!empty($password_new) && strlen($password_new) < 6) {
        $password_err = "Password must be at least 6 characters.";
        $valid = false;
    }
    if ($branch_id_new <= 0) {
        $branch_err = "Branch must be selected.";
        $valid = false;
    }

    if ($valid) {
        // Check for username/email uniqueness excluding current
        $stmt = $conn->prepare("SELECT user_id FROM users WHERE (username = ? OR email = ?) AND user_id <> ?");
        $stmt->bind_param("ssi", $username_new, $email_new, $id);
        $stmt->execute();
        $stmt->store_result();
        if ($stmt->num_rows > 0) {
            $username_err = "Username or Email already taken.";
            $stmt->close();
        } else {
            $stmt->close();

            if (!empty($password_new)) {
                $hashed_password = password_hash($password_new, PASSWORD_DEFAULT);
                $update = $conn->prepare("UPDATE users SET username=?, full_name=?, email=?, role=?, password_hash=?, branch_id=? WHERE user_id=?");
                $update->bind_param("ssssiii", $username_new, $full_name_new, $email_new, $role_new, $hashed_password, $branch_id_new, $id);
            } else {
                $update = $conn->prepare("UPDATE users SET username=?, full_name=?, email=?, role=?, branch_id=? WHERE user_id=?");
                $update->bind_param("ssssii", $username_new, $full_name_new, $email_new, $role_new, $branch_id_new, $id);
            }
            if ($update->execute()) {
                $success_message = "User updated successfully.";
                // Update local variables for form redisplay
                $username = $username_new;
                $full_name = $full_name_new;
                $email = $email_new;
                $role = $role_new;
                $branch_id = $branch_id_new;
            } else {
                echo "Error updating user.";
            }
            $update->close();
        }
    }
}
$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Edit User - Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
    <style>.wrapper{max-width:500px; margin:30px auto;}</style>
</head>
<body>
<div class="wrapper">
    <h2>Edit User</h2>
    <?php if ($success_message): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success_message) ?></div>
    <?php endif; ?>
    <form method="post" action="edit_user.php?id=<?= $id ?>">
        <div class="form-group mb-3">
            <label>Username</label>
            <input type="text" name="username" class="form-control <?= $username_err ? 'is-invalid' : '' ?>" value="<?= htmlspecialchars($username) ?>" />
            <div class="invalid-feedback"><?= htmlspecialchars($username_err) ?></div>
        </div>
        <div class="form-group mb-3">
            <label>Full Name</label>
            <input type="text" name="full_name" class="form-control" value="<?= htmlspecialchars($full_name) ?>" />
        </div>
        <div class="form-group mb-3">
            <label>Email</label>
            <input type="email" name="email" class="form-control <?= $email_err ? 'is-invalid' : '' ?>" value="<?= htmlspecialchars($email) ?>" />
            <div class="invalid-feedback"><?= htmlspecialchars($email_err) ?></div>
        </div>
        <div class="form-group mb-3">
            <label>Role</label>
            <select name="role" class="form-control" required>
                <option value="sales_rep" <?= ($role === 'sales_rep') ? 'selected' : '' ?>>Sales Rep</option>
                <option value="admin" <?= ($role === 'admin') ? 'selected' : '' ?>>Admin</option>
            </select>
        </div>
        <div class="form-group mb-3">
            <label>Branch</label>
            <select name="branch_id" class="form-control <?= isset($branch_err) ? 'is-invalid' : '' ?>" required>
                <option value="">Select branch</option>
                <?php foreach ($branches as $branch): ?>
                    <option value="<?= $branch['branch_id'] ?>" <?= ($branch['branch_id'] == $branch_id) ? 'selected' : '' ?>>
                        <?= htmlspecialchars($branch['branch_name']) ?>
                    </option>
                <?php endforeach; ?>
            </select>
            <?php if (isset($branch_err)): ?>
                <div class="invalid-feedback"><?= htmlspecialchars($branch_err) ?></div>
            <?php endif; ?>
        </div>
        <div class="form-group mb-3">
            <label>New Password (leave blank to keep current)</label>
            <input type="password" name="password" class="form-control <?= $password_err ? 'is-invalid' : '' ?>" />
            <div class="invalid-feedback"><?= htmlspecialchars($password_err) ?></div>
        </div>
        <button type="submit" class="btn btn-primary">Update User</button>
        <a href="manage_users.php" class="btn btn-secondary">Cancel</a>
    </form>
</div>
</body>
</html>
