<?php
session_start();
require_once "../includes/role_helpers.php";
requireAdmin();
require_once "../config/config.php";
require_once __DIR__ . '/vendor/autoload.php';

use Picqer\Barcode\BarcodeGeneratorPNG;

$conn = getDbConnection();
$generator = new BarcodeGeneratorPNG();

// Fetch branches for branch select dropdown
$branch_result = $conn->query("SELECT branch_id, branch_name FROM branches ORDER BY branch_name");
$branches = $branch_result ? $branch_result->fetch_all(MYSQLI_ASSOC) : [];

// Fetch categories for dropdown
$category_result = $conn->query("SELECT category_name FROM categories ORDER BY category_name");
$categories = $category_result ? $category_result->fetch_all(MYSQLI_ASSOC) : [];

$success_msg = "";
$error_msg = "";

// Handle product (inventory) add/update/delete
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $product_id = $_POST['product_id'] ?? null;
    $name = trim($_POST['product_name'] ?? '');
    // New wholesale price input
    $wholesale_price = floatval($_POST['wholesale_price'] ?? 0);
    // Calculate price as wholesale_price * 1.4
    $price = round($wholesale_price * 1.4, 2);
    $category = trim($_POST['category'] ?? '');
    $barcode = trim($_POST['barcode'] ?? '');
    $stock_quantity = intval($_POST['stock_quantity'] ?? 0);
    $expiry_date = $_POST['expiry_date'] ?? null;
    $batch_number = trim($_POST['batch_number'] ?? '');
    $branch_id = intval($_POST['branch_id'] ?? 0);

    if ($name && $wholesale_price > 0 && $branch_id > 0 && $category) {
        if ($product_id) {
            // Update including wholesale_price & updated price
            $stmt = $conn->prepare("UPDATE products SET product_name = ?, wholesale_price = ?, price = ?, category = ?, barcode = ?, stock_quantity = ?, expiry_date = ?, batch_number = ?, branch_id = ? WHERE product_id = ?");
            $stmt->bind_param("sddssissii", $name, $wholesale_price, $price, $category, $barcode, $stock_quantity, $expiry_date, $batch_number, $branch_id, $product_id);
            if ($stmt->execute()) {
                $success_msg = "Inventory item updated successfully.";
            } else {
                $error_msg = "Error updating inventory item: " . $stmt->error;
            }
            $stmt->close();
        } else {
            // Insert including wholesale_price & calculated price
            $stmt = $conn->prepare("INSERT INTO products (product_name, wholesale_price, price, category, barcode, stock_quantity, expiry_date, batch_number, branch_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
            $stmt->bind_param("sddssissi", $name, $wholesale_price, $price, $category, $barcode, $stock_quantity, $expiry_date, $batch_number, $branch_id);
            if ($stmt->execute()) {
                $success_msg = "Inventory item added successfully.";
            } else {
                $error_msg = "Error adding inventory item: " . $stmt->error;
            }
            $stmt->close();
        }
    } else {
        $error_msg = "Please fill all required fields with valid data including branch and category selection, and provide a wholesale price.";
    }
}

if (isset($_GET['delete'])) {
    $del_id = intval($_GET['delete']);
    $stmt = $conn->prepare("DELETE FROM products WHERE product_id = ?");
    $stmt->bind_param("i", $del_id);
    if ($stmt->execute()) {
        $success_msg = "Inventory item deleted successfully.";
    } else {
        $error_msg = "Error deleting inventory item: " . $stmt->error;
    }
    $stmt->close();
}

// Use branch_id from session filter; admins (branch_id=0) see all
$filter_branch_id = $_SESSION['branch_id'] ?? 0;
if ($filter_branch_id === 0) {
    $sql = "SELECT product_id, product_name, wholesale_price, price, barcode, category, stock_quantity, expiry_date, batch_number, branch_id FROM products ORDER BY product_name";
    $stmt = $conn->prepare($sql);
} else {
    $sql = "SELECT product_id, product_name, wholesale_price, price, barcode, category, stock_quantity, expiry_date, batch_number, branch_id FROM products WHERE branch_id = ? ORDER BY product_name";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $filter_branch_id);
}
$stmt->execute();
$products_result = $stmt->get_result();

?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Inventory Management - Admin</title>
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
    <style>
        /* Your existing styling */
    </style>
</head>
<body>
<div class="container">
    <a href="dashboard.php" class="btn btn-secondary mb-3">&larr; Back to Dashboard</a>
    <h2>Inventory Management</h2>
    <?php if ($success_msg): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success_msg) ?></div>
    <?php elseif ($error_msg): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($error_msg) ?></div>
    <?php endif; ?>

    <form method="post" class="row g-3 align-items-center">
        <input type="hidden" name="product_id" id="product_id" />
        <div class="col-md-3">
            <input type="text" name="product_name" id="product_name" placeholder="Product Name" class="form-control" required />
        </div>
        <div class="col-md-2">
            <input type="number" name="wholesale_price" id="wholesale_price" placeholder="Wholesale Price" step="0.01" min="0" class="form-control" required />
        </div>
        <div class="col-md-2">
            <input type="number" name="price" id="price" placeholder="Price (Auto)" class="form-control" readonly />
        </div>
        <div class="col-md-2">
            <select name="category" id="category" class="form-select" required>
                <option value="" disabled selected>-- Select Category --</option>
                <?php foreach ($categories as $cat): ?>
                    <option value="<?= htmlspecialchars($cat['category_name']) ?>"><?= htmlspecialchars($cat['category_name']) ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-md-2">
            <input type="text" name="barcode" id="barcode" placeholder="Barcode" class="form-control" />
        </div>
        <div class="col-md-1">
            <input type="number" name="stock_quantity" id="stock_quantity" placeholder="Stock" min="0" class="form-control" />
        </div>
        <div class="col-md-2">
            <input type="date" name="expiry_date" id="expiry_date" class="form-control" />
        </div>
        <div class="col-md-2">
            <input type="text" name="batch_number" id="batch_number" placeholder="Batch #" class="form-control" />
        </div>
        <div class="col-md-3">
            <select name="branch_id" id="branch_id" class="form-select" required>
                <option value="">Select Branch</option>
                <?php foreach ($branches as $branch): ?>
                    <option value="<?= $branch['branch_id'] ?>"><?= htmlspecialchars($branch['branch_name']) ?></option>
                <?php endforeach; ?>
            </select>
        </div>
        <div class="col-md-1 d-grid">
            <button type="submit" class="btn btn-primary">Save</button>
        </div>
    </form>

    <table class="table mt-4">
        <thead>
            <tr>
                <th>Name</th>
                <th>Wholesale Price (₵)</th>
                <th>Selling Price (₵)</th>
                <th>Category</th>
                <th>Barcode</th>
                <th>Stock</th>
                <th>Expiry</th>
                <th>Batch</th>
                <th>Barcode Image</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
        <?php while ($prod = $products_result->fetch_assoc()): ?>
            <tr>
                <td data-label="Name"><?= htmlspecialchars($prod['product_name']) ?></td>
                <td data-label="Wholesale Price"><?= number_format($prod['wholesale_price'], 2) ?></td>
                <td data-label="Selling Price"><?= number_format($prod['price'], 2) ?></td>
                <td data-label="Category"><?= htmlspecialchars($prod['category']) ?></td>
                <td data-label="Barcode"><?= htmlspecialchars($prod['barcode']) ?></td>
                <td data-label="Stock"><?= intval($prod['stock_quantity']) ?></td>
                <td data-label="Expiry"><?= htmlspecialchars($prod['expiry_date']) ?></td>
                <td data-label="Batch"><?= htmlspecialchars($prod['batch_number']) ?></td>
                <td data-label="Barcode Image">
                    <?php if (!empty($prod['barcode'])): ?>
                        <img src="data:image/png;base64,<?= base64_encode($generator->getBarcode($prod['barcode'], $generator::TYPE_CODE_128, 2, 50)) ?>" alt="Barcode" />
                    <?php else: ?>
                        N/A
                    <?php endif; ?>
                </td>
                <td data-label="Actions">
                    <button class="btn btn-sm btn-info edit-btn" 
                            data-id="<?= $prod['product_id'] ?>" 
                            data-name="<?= htmlspecialchars($prod['product_name'], ENT_QUOTES) ?>"
                            data-wholesale_price="<?= $prod['wholesale_price'] ?>"
                            data-category="<?= htmlspecialchars($prod['category'], ENT_QUOTES) ?>"
                            data-price="<?= $prod['price'] ?>"
                            data-barcode="<?= htmlspecialchars($prod['barcode'], ENT_QUOTES) ?>"
                            data-stock="<?= $prod['stock_quantity'] ?>"
                            data-expiry="<?= $prod['expiry_date'] ?>"
                            data-batch="<?= htmlspecialchars($prod['batch_number'], ENT_QUOTES) ?>"
                            data-branch="<?= $prod['branch_id'] ?>"
                    >Edit</button>
                    <a href="?delete=<?= $prod['product_id'] ?>" class="btn btn-sm btn-danger" onclick="return confirm('Delete this product?');">Delete</a>
                </td>
            </tr>
        <?php endwhile; ?>
        </tbody>
    </table>
</div>

<script>
// Auto calculate selling price when wholesale price changes in form
document.getElementById('wholesale_price').addEventListener('input', function() {
    let wholesalePrice = parseFloat(this.value) || 0;
    let price = wholesalePrice * 1.4;
    document.getElementById('price').value = price.toFixed(2);
});

// Populate form when clicking Edit button
document.querySelectorAll('.edit-btn').forEach(button => {
    button.addEventListener('click', () => {
        document.getElementById('product_id').value = button.dataset.id;
        document.getElementById('product_name').value = button.dataset.name;
        document.getElementById('wholesale_price').value = button.dataset.wholesale_price;
        // Trigger price calculation based on wholesale price
        let priceInput = document.getElementById('price');
        priceInput.value = (button.dataset.wholesale_price * 1.4).toFixed(2);

        document.getElementById('category').value = button.dataset.category;
        document.getElementById('barcode').value = button.dataset.barcode;
        document.getElementById('stock_quantity').value = button.dataset.stock;
        document.getElementById('expiry_date').value = button.dataset.expiry;
        document.getElementById('batch_number').value = button.dataset.batch;
        document.getElementById('branch_id').value = button.dataset.branch;
    });
});
</script>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

</body>
</html>
