<?php
require_once "../config/config.php";
session_start();
require_once "../includes/role_helpers.php";
requireLogin();

$conn = getDbConnection();

if (!isset($_SESSION['cart'])) {
    $_SESSION['cart'] = [];
}

$error = null;

// Fetch units for dropdown
$units_result = $conn->query("SELECT unit_name FROM units ORDER BY unit_name");
$units = $units_result ? $units_result->fetch_all(MYSQLI_ASSOC) : [];

// Fetch products for initial page load
$products_result = $conn->query("SELECT product_id, product_name, barcode, price, stock_quantity FROM products ORDER BY product_name ASC");
$products = [];
while ($pr = $products_result->fetch_assoc()) {
    $products[$pr['product_id']] = $pr;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Add product to cart
    if (isset($_POST['add_product_id'], $_POST['quantity'], $_POST['unit'])) {
        $pid = intval($_POST['add_product_id']);
        $qty = max(1, intval($_POST['quantity']));
        $unit = trim($_POST['unit']);
        if (isset($_SESSION['cart'][$pid])) {
            $_SESSION['cart'][$pid]['quantity'] += $qty;
            $_SESSION['cart'][$pid]['unit'] = $unit;
        } else {
            $_SESSION['cart'][$pid] = ['quantity' => $qty, 'unit' => $unit];
        }
        header("Location: pos.php");
        exit;
    }

    // Update cart quantities and units
    if (isset($_POST['update_cart'])) {
        foreach ($_POST['quantities'] as $pid => $qty) {
            $pid = intval($pid);
            $qty = max(0, intval($qty));
            if ($qty === 0) {
                unset($_SESSION['cart'][$pid]);
            } else {
                if (isset($_SESSION['cart'][$pid])) {
                    $_SESSION['cart'][$pid]['quantity'] = $qty;
                    if (isset($_POST['units'][$pid])) {
                        $_SESSION['cart'][$pid]['unit'] = trim($_POST['units'][$pid]);
                    }
                }
            }
        }
        header("Location: pos.php");
        exit;
    }

    // Checkout process
    if (isset($_POST['checkout'])) {
        if (empty($_SESSION['cart'])) {
            $error = "Your cart is empty. Add items before checkout.";
        } else {
            $conn->autocommit(false);
            try {
                $sales_rep_id = $_SESSION['user_id'];
                $sale_date = date('Y-m-d H:i:s');
                $total_amount = 0;

                // Validate stock and calculate total
                foreach ($_SESSION['cart'] as $pid => $item) {
                    if (!isset($products[$pid])) {
                        throw new Exception("Invalid product ID: $pid");
                    }
                    if ($item['quantity'] > $products[$pid]['stock_quantity']) {
                        throw new Exception("Insufficient stock for product: " . $products[$pid]['product_name']);
                    }
                    $total_amount += $products[$pid]['price'] * $item['quantity'];
                }

                // Insert sale record
                $stmt = $conn->prepare("INSERT INTO sales (sales_rep_id, sale_date, total_amount) VALUES (?, ?, ?)");
                if (!$stmt) {
                    throw new Exception("Prepare failed for sales: " . $conn->error);
                }
                $stmt->bind_param("isd", $sales_rep_id, $sale_date, $total_amount);
                if (!$stmt->execute()) {
                    throw new Exception("Execute failed for sales: " . $stmt->error);
                }
                $sale_id = $stmt->insert_id;
                $stmt->close();

                // Insert sale items and update stock
                $stmt_item = $conn->prepare("INSERT INTO sale_items (sale_id, product_id, quantity, unit_price, unit) VALUES (?, ?, ?, ?, ?)");
                $stmt_update_stock = $conn->prepare("UPDATE products SET stock_quantity = stock_quantity - ? WHERE product_id = ? AND stock_quantity >= ?");
                if (!$stmt_item || !$stmt_update_stock) {
                    throw new Exception("Prepare failed for sale_items or stock update: " . $conn->error);
                }

                foreach ($_SESSION['cart'] as $pid => $item) {
                    $quantity = (int)$item['quantity'];
                    $unit_price = $products[$pid]['price'];
                    $unit = $item['unit'];

                    $stmt_item->bind_param("iiids", $sale_id, $pid, $quantity, $unit_price, $unit);
                    if (!$stmt_item->execute()) {
                        throw new Exception("Sale item insert failed for product ID $pid: " . $stmt_item->error);
                    }

                    $stmt_update_stock->bind_param("iii", $quantity, $pid, $quantity);
                    if (!$stmt_update_stock->execute()) {
                        throw new Exception("Stock update failed for product ID $pid: " . $stmt_update_stock->error);
                    }
                    if ($stmt_update_stock->affected_rows === 0) {
                        throw new Exception("Insufficient stock for product ID: $pid");
                    }
                }

                $stmt_item->close();
                $stmt_update_stock->close();

                $conn->commit();
                $conn->autocommit(true);

                $_SESSION['cart'] = [];
                $_POST = [];

                header("Location: pos_success.php?sale_id=$sale_id");
                exit;
            } catch (Exception $e) {
                $conn->rollback();
                $conn->autocommit(true);
                $error = "Checkout failed: " . htmlspecialchars($e->getMessage());
            }
        }
    }
}

// Prepare cart items for display
$cart_items = [];
if (!empty($_SESSION['cart'])) {
    $ids = implode(',', array_map('intval', array_keys($_SESSION['cart'])));
    $sql = "SELECT product_id, product_name, price FROM products WHERE product_id IN ($ids)";
    $result = $conn->query($sql);
    while ($row = $result->fetch_assoc()) {
        $pid = $row['product_id'];
        $item = $_SESSION['cart'][$pid];
        $cart_items[$pid] = [
            'product_name' => $row['product_name'],
            'price' => $row['price'],
            'quantity' => $item['quantity'],
            'unit' => $item['unit'],
            'subtotal' => $row['price'] * $item['quantity'],
        ];
    }
}

$total = 0;
foreach ($cart_items as $item) {
    $total += $item['subtotal'];
}

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<title>POS - Adwenpa Pharma</title>
<meta name="viewport" content="width=device-width, initial-scale=1" />
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
</head>
<body>
<nav class="navbar navbar-expand-lg navbar-dark bg-primary">
<div class="container-fluid">
<a class="navbar-brand" href="dashboard.php">Adwenpa Pharma</a>
<div class="d-flex">
<a href="dashboard.php" class="btn btn-secondary me-2">Back to Dashboard</a>
<a href="logout.php" class="btn btn-outline-light btn-sm">Logout</a>
</div>
</div>
</nav>

<div class="container my-4">
<h2>Point of Sale</h2>

<?php if ($error): ?>
<div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
<?php endif; ?>

<div class="row g-3 mb-3 align-items-center">
<div class="col-auto">
<label for="barcode_input" class="col-form-label">Scan Barcode</label>
</div>
<div class="col-auto" style="max-width: 250px;">
<input type="text" id="barcode_input" class="form-control" placeholder="Scan barcode here" autofocus autocomplete="off" />
</div>
</div>

<form method="post" id="pos_form" class="row g-3 mb-4 align-items-center">
<div class="col-auto">
<label for="product_search" class="col-form-label">Search Product</label>
</div>
<div class="col-auto" style="max-width: 300px;">
<input type="text" id="product_search" class="form-control" placeholder="Type to search..." autocomplete="off" />
</div>

<div class="col-auto">
<label for="add_product_id" class="col-form-label">Product</label>
</div>
<div class="col-auto">
<select name="add_product_id" id="add_product_id" class="form-select" required>
<option value="" disabled selected>Select product</option>
<?php foreach ($products as $id => $p): ?>
<option value="<?= $id ?>" <?= (isset($_POST['add_product_id']) && $_POST['add_product_id'] == $id) ? 'selected' : '' ?>>
<?= htmlspecialchars($p['product_name']) ?> - ₵<?= number_format($p['price'], 2) ?>
</option>
<?php endforeach; ?>
</select>
</div>

<div class="col-auto">
<label for="unit" class="col-form-label">Unit</label>
</div>
<div class="col-auto">
<select name="unit" id="unit" class="form-select" required>
<option value="" disabled <?= empty($_POST['unit']) ? 'selected' : '' ?>>Select unit</option>
<?php foreach ($units as $unit_item): ?>
<option value="<?= htmlspecialchars($unit_item['unit_name']) ?>"
<?= (isset($_POST['unit']) && $_POST['unit'] === $unit_item['unit_name']) ? 'selected' : '' ?>>
<?= htmlspecialchars($unit_item['unit_name']) ?>
</option>
<?php endforeach; ?>
</select>
</div>

<div class="col-auto">
<label for="quantity" class="col-form-label">Quantity</label>
</div>
<div class="col-auto">
<input type="number" name="quantity" id="quantity" value="<?= isset($_POST['quantity']) ? intval($_POST['quantity']) : 1 ?>" min="1" class="form-control" required />
</div>

<div class="col-auto">
<button type="submit" class="btn btn-primary">Add to Cart</button>
</div>
</form>

<?php if (empty($cart_items)): ?>
<div class="alert alert-info">Your cart is empty.</div>
<?php else: ?>
<form method="post" class="mb-0">
<table class="table table-bordered align-middle text-center">
<thead class="table-light">
<tr>
<th>Product</th>
<th>Unit</th>
<th>Price (₵)</th>
<th>Quantity</th>
<th>Subtotal (₵)</th>
</tr>
</thead>
<tbody>
<?php foreach ($cart_items as $pid => $item): ?>
<tr>
<td><?= htmlspecialchars($item['product_name']) ?></td>
<td><?= htmlspecialchars($item['unit']) ?></td>
<td><?= number_format($item['price'], 2) ?></td>
<td>
<input type="number" name="quantities[<?= $pid ?>]" value="<?= $item['quantity'] ?>" min="0" class="form-control form-control-sm" />
</td>
<td><?= number_format($item['subtotal'], 2) ?></td>
</tr>
<?php endforeach; ?>
<tr class="fw-bold">
<td colspan="4" class="text-end">Total</td>
<td>₵<?= number_format($total, 2) ?></td>
</tr>
</tbody>
</table>

<div class="d-flex justify-content-between">
<button type="submit" name="update_cart" class="btn btn-secondary">Update Cart</button>
<button type="submit" name="checkout" class="btn btn-success" onclick="return confirm('Confirm checkout?')">Checkout</button>
</div>
</form>
<?php endif; ?>
</div>

<script>
window.onload = function() {
  document.getElementById('product_search').addEventListener('input', function () {
    const query = this.value.trim();
    fetch('product_search.php?q=' + encodeURIComponent(query))
      .then(response => response.json())
      .then(products => {
        const select = document.getElementById('add_product_id');
        select.innerHTML = '';

        const defaultOption = document.createElement('option');
        defaultOption.text = 'Select product';
        defaultOption.disabled = true;
        defaultOption.selected = true;
        select.appendChild(defaultOption);

        if (products.length === 0) {
          const option = document.createElement('option');
          option.textContent = 'No products found';
          option.disabled = true;
          select.appendChild(option);
        } else {
          products.forEach(product => {
            const option = document.createElement('option');
            option.value = product.product_id;
            option.textContent = product.product_name + ' - ₵' + product.price.toFixed(2);
            select.appendChild(option);
          });
        }
      })
      .catch(error => {
        console.error('Product search error:', error);
      });
  });
};

document.getElementById('barcode_input').addEventListener('keydown', function(e) {
if (e.key === 'Enter') {
    e.preventDefault();
    const barcode = this.value.trim();
    if (barcode.length === 0) return;
    this.value = ''; // clear input

    fetch('pos_barcode_lookup.php?barcode=' + encodeURIComponent(barcode))
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const form = document.createElement('form');
                form.method = 'POST';
                form.action = 'pos.php';

                const inputId = document.createElement('input');
                inputId.type = 'hidden';
                inputId.name = 'add_product_id';
                inputId.value = data.product_id;
                form.appendChild(inputId);

                const inputQty = document.createElement('input');
                inputQty.type = 'hidden';
                inputQty.name = 'quantity';
                inputQty.value = 1;
                form.appendChild(inputQty);

                document.body.appendChild(form);
                form.submit();
            } else {
                alert('Product not found for barcode: ' + barcode);
            }
        })
        .catch(() => {
            alert('Error looking up barcode.');
        });
}
});

// Reset form after checkout submission (if arriving fresh)
window.onload = function() {
if ('<?= isset($_POST['checkout']) ? '1' : '0' ?>' === '1') {
    const form = document.getElementById('pos_form');
    if(form) {
        form.reset();
    }
}
};
</script>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
