<?php
require_once "../config/config.php";
session_start();
require_once "../includes/role_helpers.php";
requireAdmin(); // ensure only admin access

$conn = getDbConnection();

$filter = $_GET['filter'] ?? 'day';

// Determine date ranges for filtering
switch ($filter) {
    case 'week':
        // Monday to Sunday of current week
        $start_date = date('Y-m-d', strtotime('monday this week'));
        $end_date = date('Y-m-d', strtotime('sunday this week'));
        break;
    case 'month':
        $start_date = date('Y-m-01'); // first day of current month
        $end_date = date('Y-m-t');    // last day of current month
        break;
    case 'day':
    default:
        $start_date = date('Y-m-d');
        $end_date = date('Y-m-d');
        break;
}

// Fetch total sales per staff within the date range
$query = "SELECT sales_rep_id, DATE(sale_date) AS sale_day, SUM(total_amount) AS total_sales
          FROM sales
          WHERE sale_date >= ? AND sale_date <= ?
          GROUP BY sales_rep_id, sale_day
          ORDER BY sale_day DESC";

$stmt = $conn->prepare($query);
$stmt->bind_param("ss", $start_date, $end_date);
$stmt->execute();
$result = $stmt->get_result();

$sales_data = $result->fetch_all(MYSQLI_ASSOC);
$stmt->close();

// Fetch sales rep names for display
$sales_rep_ids = array_unique(array_column($sales_data, 'sales_rep_id'));
$staff_names = [];
if (!empty($sales_rep_ids)) {
    $ids_str = implode(',', array_map('intval', $sales_rep_ids));
    $staff_res = $conn->query("SELECT user_id, full_name FROM users WHERE user_id IN ($ids_str)");
    while ($row = $staff_res->fetch_assoc()) {
        $staff_names[$row['user_id']] = $row['full_name'];
    }
}

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<title>Staff Sales Report - Admin</title>
<link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
<meta name="viewport" content="width=device-width, initial-scale=1" />
</head>
<body>
<div class="container my-4">
<h2>Staff Sales Report</h2>

<form method="GET" class="mb-4 row g-3 align-items-center">
  <div class="col-auto">
    <label for="filter" class="col-form-label">Filter by:</label>
  </div>
  <div class="col-auto">
    <select name="filter" id="filter" class="form-select" onchange="this.form.submit()">
      <option value="day" <?= ($filter === 'day') ? 'selected' : '' ?>>Today</option>
      <option value="week" <?= ($filter === 'week') ? 'selected' : '' ?>>This Week</option>
      <option value="month" <?= ($filter === 'month') ? 'selected' : '' ?>>This Month</option>
    </select>
  </div>
</form>

<table class="table table-bordered table-striped">
  <thead>
    <tr>
      <th>Staff Name</th>
      <th>Date</th>
      <th>Total Sales (₵)</th>
    </tr>
  </thead>
  <tbody>
    <?php if (empty($sales_data)): ?>
      <tr><td colspan="3" class="text-center">No sales records found for the selected period.</td></tr>
    <?php else: ?>
      <?php foreach ($sales_data as $row): ?>
        <tr>
          <td><?= htmlspecialchars($staff_names[$row['sales_rep_id']] ?? 'Unknown') ?></td>
          <td><?= htmlspecialchars($row['sale_day']) ?></td>
          <td><?= number_format($row['total_sales'], 2) ?></td>
        </tr>
      <?php endforeach; ?>
    <?php endif; ?>
  </tbody>
</table>

</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
