<?php
session_start();
require_once "../includes/role_helpers.php";
requireAdmin();
require_once "../config/config.php";

$conn = getDbConnection();

$low_stock_threshold = 10;
$today = date('Y-m-d');
$error_msg = "";
$success_msg = "";

// Branch ID from session (must be set on login)
if (!isset($_SESSION['branch_id'])) {
    die('Branch not set in session.');
}

$branch_id = $_SESSION['branch_id'];
$is_admin = ($branch_id === 0);

// Add new stock
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    if ($_POST['action'] === 'add') {
        $product_id = intval($_POST['product_id'] ?? 0);
        $quantity = intval($_POST['quantity'] ?? 0);
        $expiry_date = $_POST['expiry_date'] ?? '';

        if ($product_id <= 0 || $quantity < 0 || !$expiry_date) {
            $error_msg = "All fields are required and quantity must be non-negative.";
        } else {
            $stock_branch_id = $is_admin ? intval($_POST['branch_id'] ?? 0) : $branch_id;
            if (!$is_admin && $stock_branch_id != $branch_id) {
                $error_msg = "You are not allowed to add stock for other branches.";
            } else {
                $stmt_check = $conn->prepare("SELECT stock_id FROM stock WHERE product_id = ? AND expiry_date = ? AND branch_id = ?");
                if (!$stmt_check) {
                    die("Prepare failed: " . $conn->error);
                }
                $stmt_check->bind_param("isi", $product_id, $expiry_date, $stock_branch_id);
                $stmt_check->execute();
                $stmt_check->store_result();
                if ($stmt_check->num_rows > 0) {
                    $error_msg = "Stock for this product, expiry date, and branch already exists. Please update it instead.";
                } else {
                    $stmt_add = $conn->prepare("INSERT INTO stock (product_id, quantity, expiry_date, branch_id) VALUES (?, ?, ?, ?)");
                    if (!$stmt_add) {
                        die("Prepare failed: " . $conn->error);
                    }
                    $stmt_add->bind_param("iisi", $product_id, $quantity, $expiry_date, $stock_branch_id);
                    if ($stmt_add->execute()) {
                        $success_msg = "Stock added successfully.";
                    } else {
                        $error_msg = "Error adding stock: " . $stmt_add->error;
                    }
                    $stmt_add->close();
                }
                $stmt_check->close();
            }
        }
    }
    if ($_POST['action'] === 'update') {
        $stock_id = intval($_POST['stock_id'] ?? 0);
        $quantity = intval($_POST['quantity'] ?? 0);
        if ($stock_id > 0 && $quantity >= 0) {
            $stmt_update = $conn->prepare("UPDATE stock SET quantity = ? WHERE stock_id = ?");
            if (!$stmt_update) {
                die("Prepare failed: " . $conn->error);
            }
            $stmt_update->bind_param("ii", $quantity, $stock_id);
            if ($stmt_update->execute()) {
                $success_msg = "Stock updated successfully.";
            } else {
                $error_msg = "Error updating stock: " . $stmt_update->error;
            }
            $stmt_update->close();
        } else {
            $error_msg = "Invalid stock update data.";
        }
    }
    if ($_POST['action'] === 'delete') {
        $stock_id = intval($_POST['stock_id'] ?? 0);
        if ($stock_id > 0) {
            $stmt_delete = $conn->prepare("DELETE FROM stock WHERE stock_id = ?");
            if (!$stmt_delete) {
                die("Prepare failed: " . $conn->error);
            }
            $stmt_delete->bind_param("i", $stock_id);
            if ($stmt_delete->execute()) {
                $success_msg = "Stock deleted successfully.";
            } else {
                $error_msg = "Error deleting stock: " . $stmt_delete->error;
            }
            $stmt_delete->close();
        } else {
            $error_msg = "Invalid stock ID for deletion.";
        }
    }
}

$search_term = trim($_GET['search'] ?? '');
$search_sql = "";
$params = [];
if ($search_term !== '') {
    $search_sql = " AND p.product_name LIKE ?";
    $params[] = "%$search_term%";
}

if ($is_admin) {
    $stmt_products = $conn->prepare("SELECT product_id, product_name FROM products ORDER BY product_name");
    if (!$stmt_products) {
        die("Prepare failed: " . $conn->error);
    }
} else {
    $stmt_products = $conn->prepare("SELECT product_id, product_name FROM products WHERE branch_id = ? ORDER BY product_name");
    if (!$stmt_products) {
        die("Prepare failed: " . $conn->error);
    }
    $stmt_products->bind_param("i", $branch_id);
}
$stmt_products->execute();
$products_res = $stmt_products->get_result();
$products = $products_res->fetch_all(MYSQLI_ASSOC);

if ($is_admin) {
    $sql = "
        SELECT s.stock_id, p.product_name, s.quantity, s.expiry_date, s.branch_id
        FROM stock AS s
        JOIN products AS p ON s.product_id = p.product_id
        WHERE 1=1 $search_sql
        ORDER BY s.quantity ASC, p.product_name ASC
    ";
    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        die("Prepare failed: " . $conn->error);
    }
    if ($search_sql) {
        $stmt->bind_param("s", $params[0]);
    }
} else {
    $sql = "
        SELECT s.stock_id, p.product_name, s.quantity, s.expiry_date, s.branch_id
        FROM stock AS s
        JOIN products AS p ON s.product_id = p.product_id
        WHERE s.branch_id = ? $search_sql
        ORDER BY s.quantity ASC, p.product_name ASC
    ";
    $stmt = $conn->prepare($sql);
    if (!$stmt) {
        die("Prepare failed: " . $conn->error);
    }
    if ($search_sql) {
        $stmt->bind_param("is", $branch_id, $params[0]);
    } else {
        $stmt->bind_param("i", $branch_id);
    }
}

$stmt->execute();
$result = $stmt->get_result();

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8" />
    <title>Stock Management - Admin</title>
    <meta name="viewport" content="width=device-width, initial-scale=1" />
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" />
    <style>
        body {
            min-height: 100vh;
            background: #f8f9fa;
            padding: 20px;
        }
        .low-stock {
            background-color: #fff3cd; /* yellow alert */
        }
        .expired {
            background-color: #f8d7da; /* red alert */
            color: #842029;
        }
    </style>
</head>
<body>
<div class="container">
    <a href="dashboard.php" class="btn btn-secondary mb-3">&larr; Back to Dashboard</a>

    <h2 class="mb-4">Stock Management</h2>

    <?php if ($error_msg): ?>
        <div class="alert alert-danger"><?= htmlspecialchars($error_msg) ?></div>
    <?php endif; ?>
    <?php if ($success_msg): ?>
        <div class="alert alert-success"><?= htmlspecialchars($success_msg) ?></div>
    <?php endif; ?>

    <div class="card mb-4 p-3 shadow-sm bg-white rounded">
        <h5>Add New Stock</h5>
        <form method="post" class="row g-3 align-items-end">
            <input type="hidden" name="action" value="add" />
            <div class="col-md-4">
                <label for="product_id" class="form-label">Product</label>
                <select class="form-select" id="product_id" name="product_id" required>
                    <option value="">Select Product</option>
                    <?php foreach ($products as $prod): ?>
                    <option value="<?= $prod['product_id'] ?>"><?= htmlspecialchars($prod['product_name']) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <?php if ($is_admin) : ?>
            <div class="col-md-3">
                <label for="branch_id" class="form-label">Branch</label>
                <select class="form-select" id="branch_id" name="branch_id" required>
                    <?php
                    $conn_tmp = getDbConnection();
                    $branches = $conn_tmp->query("SELECT branch_id, branch_name FROM branches ORDER BY branch_name")->fetch_all(MYSQLI_ASSOC);
                    $conn_tmp->close();
                    ?>
                    <option value="">Select Branch</option>
                    <?php foreach ($branches as $branch) : ?>
                    <option value="<?= $branch['branch_id'] ?>"><?= htmlspecialchars($branch['branch_name']) ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <?php endif; ?>
            <div class="col-md-3">
                <label for="quantity" class="form-label">Quantity</label>
                <input type="number" id="quantity" name="quantity" min="0" class="form-control" required />
            </div>
            <div class="col-md-2">
                <label for="expiry_date" class="form-label">Expiry Date</label>
                <input type="date" id="expiry_date" name="expiry_date" class="form-control" required />
            </div>
            <div class="col-md-1 d-grid">
                <button type="submit" class="btn btn-primary">Add</button>
            </div>
        </form>
    </div>

    <form method="get" class="mb-3 d-flex gap-2">
        <input type="search" name="search" class="form-control" placeholder="Search product name..." value="<?= htmlspecialchars($search_term) ?>" />
        <button type="submit" class="btn btn-primary">Search</button>
        <a href="stock_management.php" class="btn btn-secondary">Reset</a>
    </form>

    <table class="table table-bordered table-hover align-middle bg-white shadow-sm rounded">
        <thead class="table-light">
            <tr>
                <th>#</th>
                <th>Product Name</th>
                <th>Quantity</th>
                <th>Expiry Date</th>
                <?php if ($is_admin) : ?>
                <th>Branch</th>
                <?php endif; ?>
                <th>Update Quantity</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php
            $row_num = 1;
            while ($row = $result->fetch_assoc()):
                $is_low_stock = $row['quantity'] < $low_stock_threshold;
                $is_expired = $row['expiry_date'] <= $today;
                $row_class = $is_expired ? 'expired' : ($is_low_stock ? 'low-stock' : '');
            ?>
            <tr class="<?= $row_class ?>">
                <td><?= $row_num++ ?></td>
                <td><?= htmlspecialchars($row['product_name']) ?></td>
                <td><?= (int)$row['quantity'] ?></td>
                <td><?= htmlspecialchars($row['expiry_date']) ?></td>
                <?php if ($is_admin): ?>
                <td><?= htmlspecialchars($row['branch_id']) ?></td>
                <?php endif; ?>
                <td>
                    <form method="post" class="d-flex gap-1 align-items-center m-0">
                        <input type="hidden" name="action" value="update" />
                        <input type="hidden" name="stock_id" value="<?= $row['stock_id'] ?>" />
                        <input type="number" name="quantity" min="0" class="form-control form-control-sm" style="width: 80px;" value="<?= (int)$row['quantity'] ?>" required />
                        <button type="submit" class="btn btn-sm btn-success">Update</button>
                    </form>
                </td>
                <td>
                    <form method="post" onsubmit="return confirm('Confirm delete this stock item?')" class="m-0">
                        <input type="hidden" name="action" value="delete" />
                        <input type="hidden" name="stock_id" value="<?= $row['stock_id'] ?>" />
                        <button type="submit" class="btn btn-sm btn-danger">Delete</button>
                    </form>
                </td>
            </tr>
            <?php endwhile;
            if ($row_num === 1): ?>
            <tr><td colspan="<?= $is_admin ? '7' : '6' ?>" class="text-center">No stock found.</td></tr>
            <?php endif; ?>
        </tbody>
    </table>
</div>
</body>
</html>
